/*********************************************************************
*	 Project:MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=12MHz, SysCLK=12MHz
*	Description
*			IAP	test
*	Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	ISP_START_ADDRESS	 0x7E00	 //ISP Start address
#define	IAP_START_ADDRESS	 0x7600	 //IAP Start address
#define	IAP_END_ADDRESS		 ISP_START_ADDRESS

#define	IAP_ERR_RETRY_CNT_MAX	5	// IAP Try again times

idata u8 TrapFlag[3];


#define	IAP_SYS_PARA_FLAG	0xA5
#define	SYS_PARA_ADDR		(IAP_START_ADDRESS+0)

//
typedef	struct{
	u16	Para1;			//
	u32	Para2;			//
	u8 Para3;
	u8 Flag;			// 0xA5	vaild flag
}SysParaDef;


typedef	union
{
	u8 BUF[sizeof(SysParaDef)];
	SysParaDef B;
}SysParaTypeDef;
xdata SysParaTypeDef SysPara;		// System para


void ReadSysParaFromIAP(void);
u8 SaveSysParaToIAP(void);
void RestoreSysPara(void);




/*************************************************
*Function:		char putchar (char c)
*Description:	send for printf
*Input:			char c
*Output:
*************************************************/
char putchar (char c)
{
	bit	bES;
	bES=ES0;
	ES0=0;
	S0BUF =	c;
	while(TI0==0);
	TI0=0;
	ES0=bES;
	return 0;
}

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void CheckTrapFlag(void)
*Description:	Check the IAP operation	whether	is Right
*				If TrapFlag	!= Win,then	the	IAP	operation is wrong,MCU reset
*Input:
*Output:
*************************************************************************************/
void CheckTrapFlag(void)
{
	if ((TrapFlag[0]!='W')||(TrapFlag[1]!='i')||(TrapFlag[2]!='n'))
	{
		//MCU Reset
		POW_ResetToAP();
	}
}

/***********************************************************************************
*Function:		u8 IAP_ReadPPage(u8	PsfrAddr)
*Description:	Read P page	sfr
*Input:
*				u8 PsfrAddr: sfr Address
*Output:
*				u8 : sfr data
*************************************************************************************/
u8 IAP_ReadPPage(u8	PsfrAddr)
{
	bit	bEA=EA;
	EA = 0;						//Turn off interrupt
	IFADRH = 0;					//IFADRH must be 0
	IFADRL=	PsfrAddr;			//sfr Address
	IFMT = ISP_READ_P;
	ISPCR =	0x80;				//Enable ISP/IAP
	CheckTrapFlag();
	SCMD = 0x46;
	CheckTrapFlag();
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR =	0;					//clear
	EA = bEA;					//restore interrupt
	return IFD;
}

/***********************************************************************************
*Function:		void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
*Description:	write P	page sfr
*Input:
*				u8 PsfrAddr: sfr Address
*				u8 PsfrData: sfr data
*Output:
*************************************************************************************/
void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
{
	bit	bEA=EA;
	EA = 0;						//Turn off interrupt
	IFADRH = 0;					//IFADRH must be 0
	IFADRL=	PsfrAddr;			//sfr Address
	IFD= PsfrData;				//sfr data
	IFMT = ISP_WRITE_P;
	ISPCR =	0x80;				//Enable ISP/IAP
	CheckTrapFlag();
	SCMD = 0x46;
	CheckTrapFlag();
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR =	0;					//clear
	EA = bEA;					//restore interrupt
}

/***********************************************************************************
*Function: u8	IAP_ReadByte(u16 ByteAddr)
*Description:	read one byte from IAP
*Input:
*				u16	ByteAddr: IAP Address
*Output:		u8:	the	data from IAP
*************************************************************************************/
u8 IAP_ReadByte(u16	ByteAddr)
{
		bit	bEA=EA;
		IFADRH = ByteAddr>>8;		//IAP Address High byte
		IFADRL=	ByteAddr;			//IAP Address Low byte
		EA = 0;						//Turn off interrupt
		IFMT = ISP_READ;
		ISPCR =	0x80;				//Enable ISP/IAP
		CheckTrapFlag();
		SCMD = 0x46;
		CheckTrapFlag();
		SCMD = 0xB9;
		IFMT=0;
		ISPCR =	0;					//clear
		EA = bEA;					//restore interrupt
		return IFD;
}
// Can use MOVC	to read	IAP
#define	IAP_ReadByteByMOVC(x)	CBYTE[x]

/***********************************************************************************
*Function:		void IAP_ErasePage(u8 ByteAddr)
*Description:	Erase one page
*				Erasing	a page takes about 30ms
*Input:			u8 ByteAddr: IAP Address High byte
*Output:
*************************************************************************************/
void IAP_ErasePage(u8 ByteAddr)
{
		bit	bEA=EA;
		IFADRH = ByteAddr;		//IAP Address High byte
		IFADRL=	0x00;			//IAP Address Low byte
		EA = 0;					//Turn off interrupt
		IFMT = ISP_ERASE;
		ISPCR =	0x80;			//Enable ISP/IAP
		CheckTrapFlag();
		SCMD = 0x46;
		CheckTrapFlag();
		SCMD = 0xB9;
		nop();
		IFMT=0;
		ISPCR =	0;					//clear
		EA = bEA;					//restore interrupt
}

/***********************************************************************************
*Function:		void IAP_WriteByte(u16 ByteAddr,u8 ByteData)
*Description:	write one byte to IAP
*				Write a	page takes about 80us
*Input:			u16	ByteAddr: IAP Address
*				u8 ByteData: the data to be	write
*Output:
*************************************************************************************/
void IAP_WriteByte(u16 ByteAddr,u8 ByteData)
{
		bit	bEA=EA;
		IFD	= ByteData;				//data to be write
		IFADRH = ByteAddr>>8;		//IAP address high
		IFADRL=	ByteAddr;			//IAP address low
		EA = 0;						//Turn off interrupt
		IFMT = ISP_WRITE;
		ISPCR =	0x80;				//Enable ISP/IAP
		CheckTrapFlag();
		SCMD = 0x46;
		CheckTrapFlag();
		SCMD = 0xB9;
		nop();
		IFMT=0;
		ISPCR =	0;					//clear
		EA = bEA;					//restore interrupt
}

/***********************************************************************************
*Function:		void InitUart0_S0BRG()
*Description:	Initialize Uart0, The source of	baud rate was S0BRG
*Input:
*Output:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R.	source:	S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source: SYSCLK
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_12000000_1T);

	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// Set P22,P24,P26 as push-pull(LED)
	PORT_SetP3QuasiBi(BIT0|BIT1);					// Set P30,P31 as Quasi-Bi
	PORT_SetP1OpenDrainPu(BIT0|BIT1);				// Set P10,P11 as open-drain with pull-high	for	digital	input
	P10=1;
	P11=1;
}


/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}


/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitUart0_S0BRG();
}

void main()
{
	u8 i;

	TrapFlag[0]='W';

	InitSystem();

	IO_LED_G=0;IO_LED_Y=0;IO_LED_R=0;
	DelayXms(1000);
	printf("\nStart	IAP	DEMO!");
	IO_LED_G=1;IO_LED_Y=1;IO_LED_R=1;

	TrapFlag[1]='i';
	TrapFlag[2]='n';
	// read	IAP	low	boundary
	i=IAP_ReadPPage(IAPLB_P);
	printf("\nRead IAPLB:%02bX",i);

	// set IAP low boundary	to ISP address,IAP size	none
	IAP_WritePPage(IAPLB_P,IAP_END_ADDRESS/256);
	i=IAP_ReadPPage(IAPLB_P);
	printf("\nNOW IAPLB:%02bX  ",i);
	TrapFlag[1]=0xFF;
	TrapFlag[2]=0xFF;


	ReadSysParaFromIAP();
	printf("\nRead SysPara:	");
	for(i=0;i<(sizeof(SysParaDef));i++)
	{
		printf("%02bX ",SysPara.BUF[i]);
	}

	printf("\nWrite	cnt:%ld",SysPara.B.Para2);

	while(1)
	{
		SysPara.B.Para1++;
		IO_LED_G=!IO_LED_G;
		DelayXms(100);
		if(P10==0)
		{
			//
			i=0;
			do{
				DelayXus(100);
				if(P10==0)
				{
					i++;
				}
				else
				{
					break;
				}
			}while (i<200);

			if(i>=200)
			{
				RestoreSysPara();
				TrapFlag[1]='i';
				printf("\nSave SysPara:	");
				for(i=0;i<(sizeof(SysParaDef));i++)
				{
					printf("%02bX ",SysPara.BUF[i]);
				}
				IO_LED_Y=1;IO_LED_R=1;
				if(SaveSysParaToIAP()==0)
				{ // Pass
					printf(" ---OK!");
					IO_LED_Y=0;
				}
				else
				{ // Fail
					printf(" ---Fail!");
					IO_LED_R=0;
					while(1)
					{
						CLRWDT();
					}
				}
				TrapFlag[1]=0xFF;

				//
				i=0;
				do{
					DelayXus(100);
					if(P10==1)
					{
						i++;
					}
					else
					{
						i=0;
					}
				}while (i<200);
			}
		}



		if(P11==0)
		{
			//
			i=0;
			do{
				DelayXus(100);
				if(P11==0)
				{
					i++;
				}
				else
				{
					break;
				}
			}while (i<200);

			if(i>=200)
			{
				SysPara.B.Para2++;
				TrapFlag[1]='i';
				printf("\nSave SysPara:	");
				for(i=0;i<(sizeof(SysParaDef));i++)
				{
					printf("%02bX ",SysPara.BUF[i]);
				}
				IO_LED_Y=1;IO_LED_R=1;
				if(SaveSysParaToIAP()==0)
				{ // Pass
					printf(" ---OK!");
					IO_LED_Y=0;
				}
				else
				{ // Fail
					printf(" ---Fail!");
					IO_LED_R=0;
					while(1)
					{
						CLRWDT();
					}
				}
				TrapFlag[1]=0xFF;

			}
		}

	}

}

/***********************************************************************************
*Function:		void ReadSysParaFromIAP(void)
*Description:	Read system	parameter from IAP
*Input:
*Output:
*************************************************************************************/
void ReadSysParaFromIAP(void)
{
	u8 i;
	u16	wAddr;
	wAddr=0;
	for(i=0;i<(512/(sizeof(SysParaDef)));i++)
	{ // looking for vaild system parameter
		if(CBYTE[(SYS_PARA_ADDR)+(sizeof(SysParaDef)-1)+wAddr]==IAP_SYS_PARA_FLAG)
		{	// find	vaild system parameter
			break;
		}
		wAddr=wAddr+(sizeof(SysParaDef));
	}
	if(i<(512/(sizeof(SysParaDef))))
	{ // find vaild	system parameter
		for(i=0;i<(sizeof(SysParaDef));i++)
		{ // read system parameter
			SysPara.BUF[i]=	CBYTE[(SYS_PARA_ADDR)+wAddr+i];
		}
	}
	else
	{ // no	vaild system parameter,restore
		RestoreSysPara();
	}
}

/***********************************************************************************
*Function:		void SaveSysParaToIAP(void)
*Description:	Write system parameter to IAP
*Input:
*Output:		0->success
*							1->fail
*************************************************************************************/
u8 SaveSysParaToIAP(void)
{
	u8 i,l,ErrCnt;
	u16	wAddr;
	bit	bWrite;
	bit	bErr;

	SysPara.B.Flag=IAP_SYS_PARA_FLAG;
	ErrCnt=0;
IAP_ERR_RETRY:
	if(ErrCnt>=IAP_ERR_RETRY_CNT_MAX) return ErrCnt;
	ErrCnt++;
	bErr=FALSE;
	bWrite=FALSE;
	TrapFlag[2]='n';
	wAddr=0;
	for(i=0;i<(512/(sizeof(SysParaDef)));i++)
	{ // looking for vaild system parameter
		if(CBYTE[(SYS_PARA_ADDR)+(sizeof(SysParaDef)-1)+wAddr]==IAP_SYS_PARA_FLAG)
		{
			break;
		}
		wAddr=wAddr+(sizeof(SysParaDef));
	}
	if(i<(512/(sizeof(SysParaDef))))
	{ // find vaild	system parameter

		for(i=0;i<sizeof(SysParaDef);i++)
		{
			wAddr++;
			if(CBYTE[(SYS_PARA_ADDR)+wAddr]!=SysPara.BUF[i])
			{
				// different from stored data before,set flag to write data	into flash
				bWrite=TRUE;
				break;
			}
		}
	}
	else
	{//	Not	find
		// set flag	to write data into flash
		bWrite=TRUE;
	}

	if(bWrite==TRUE)
	{
		// set IAP start address
		IAP_WritePPage(IAPLB_P,IAP_START_ADDRESS>>8);

		// write data into flash
		wAddr=0;
		for(i=0;i<(512/(sizeof(SysParaDef)));i++)
		{
			// // checking data	whether	is empty
			for(l=0;l<(sizeof(SysParaDef));l++)
			{
				if(CBYTE[(SYS_PARA_ADDR)+wAddr+l]!=0xFF)
				{
					break;
				}

			}
			if(l<(sizeof(SysParaDef)))
			{
				// not empty,clear flag
				IAP_WriteByte((SYS_PARA_ADDR)+(sizeof(SysParaDef)-1)+wAddr,	0x00);
			}
			else
			{ // empty

				// write data into flash
				for(l=0;l<sizeof(SysParaDef);l++)
				{
					IAP_WriteByte((SYS_PARA_ADDR)+wAddr+l, SysPara.BUF[l]);
				}
				break;
			}
			wAddr=wAddr+(sizeof(SysParaDef));
		}
		if(i<(512/(sizeof(SysParaDef))))
		{
			// data	has	been written into flash
		}
		else
		{
			// not found
			wAddr=0x0000;
			// erase page
			IAP_ErasePage(SYS_PARA_ADDR>>8);
			// write data into flash
			for(l=0;l<sizeof(SysParaDef);l++)
			{
				IAP_WriteByte((SYS_PARA_ADDR)+l, SysPara.BUF[l]);
			}
		}
		// reset IAP start address
		IAP_WritePPage(IAPLB_P,IAP_END_ADDRESS>>8);

		// check data whether is OK
		for(l=0;l<(sizeof(SysParaDef));l++)
		{
			if(CBYTE[(SYS_PARA_ADDR)+wAddr+l]!=	SysPara.BUF[l])
			{
				bErr=TRUE;
				break;
			}

		}

	}

	TrapFlag[2]=0xFF;

	if(bErr==TRUE) goto	IAP_ERR_RETRY;

	return 0;

}
/***********************************************************************************
*Function:		void RestoreSysPara(void)
*Description:	Restore	system parameter
*Input:
*Output:
*************************************************************************************/
void RestoreSysPara(void)
{
	// Restore system parameter
	SysPara.B.Para1=0x0000;
	SysPara.B.Para2=0x00000000;
	SysPara.B.Para3=0x00;
	SysPara.B.Flag=IAP_SYS_PARA_FLAG;
}




